<?php

namespace Notif;

use Firebase\JWT\JWT;
use Firebase\JWT\Key;

class APNSPushNotification
{
    private $teamId;
    private $keyId;
    private $privateKeyPath;
    private $bundleId;
    private $isProduction;
    
    public function __construct($teamId, $keyId, $privateKeyPath, $bundleId, $isProduction = false)
    {
        $this->teamId = $teamId;
        $this->keyId = $keyId;
        $this->privateKeyPath = $privateKeyPath;
        $this->bundleId = $bundleId;
        $this->isProduction = $isProduction;
    }
    
    /**
     * Generate JWT token untuk autentikasi APNs
     */
    private function generateAuthToken()
    {
        $header = [
            'alg' => 'ES256',
            'kid' => $this->keyId
        ];
        
        $payload = [
            'iss' => $this->teamId,
            'iat' => time()
        ];
        
        if (!file_exists($this->privateKeyPath)) {
            throw new \Exception("Private key file tidak ditemukan: " . $this->privateKeyPath);
        }
        
        $privateKey = file_get_contents($this->privateKeyPath);
        
        return JWT::encode($payload, $privateKey, 'ES256', null, $header);
    }
    
    /**
     * Kirim notifikasi ke device token
     */
    public function sendNotification($deviceToken, $title, $body, $data = [], $options = [])
    {
        $authToken = $this->generateAuthToken();
        
        // Endpoint APNs
        $endpoint = $this->isProduction ? 
            'https://api.push.apple.com' : 
            'https://api.sandbox.push.apple.com';
        
        $url = $endpoint . '/3/device/' . $deviceToken;
        
        // Default payload
        $payload = [
            'aps' => [
                'alert' => [
                    'title' => $title,
                    'body' => $body
                ],
                'sound' => $options['sound'] ?? 'default',
                'badge' => $options['badge'] ?? 1
            ]
        ];
        
        // Tambahkan custom data jika ada
        if (!empty($data)) {
            $payload = array_merge($payload, $data);
        }
        
        // Tambahkan options tambahan
        if (isset($options['category'])) {
            $payload['aps']['category'] = $options['category'];
        }
        
        if (isset($options['thread_id'])) {
            $payload['aps']['thread-id'] = $options['thread_id'];
        }
        
        // Headers untuk request
        $headers = [
            'apns-topic: ' . $this->bundleId,
            'apns-push-type: alert',
            'authorization: bearer ' . $authToken,
            'content-type: application/json'
        ];
        
        // Tambahkan apns-priority jika di-set
        if (isset($options['priority'])) {
            $headers[] = 'apns-priority: ' . $options['priority'];
        }
        
        // Tambahkan apns-expiration jika di-set
        if (isset($options['expiration'])) {
            $headers[] = 'apns-expiration: ' . $options['expiration'];
        }
        
        // Kirim request menggunakan cURL
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTP_VERSION, CURL_HTTP_VERSION_2_0);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        
        curl_close($ch);
        
        return [
            'success' => $httpCode === 200,
            'http_code' => $httpCode,
            'response' => $response,
            'error' => $error,
            'payload' => $payload,
            'headers' => $headers
        ];
    }
    
    /**
     * Kirim notifikasi ke multiple devices
     */
    public function sendToMultipleDevices($deviceTokens, $title, $body, $data = [], $options = [])
    {
        $results = [];
        
        foreach ($deviceTokens as $deviceToken) {
            $results[] = $this->sendNotification($deviceToken, $title, $body, $data, $options);
        }
        
        return $results;
    }
    
    /**
     * Kirim notifikasi dengan custom payload
     */
    public function sendCustomNotification($deviceToken, $customPayload, $options = [])
    {
        $authToken = $this->generateAuthToken();
        
        $endpoint = $this->isProduction ? 
            'https://api.push.apple.com' : 
            'https://api.sandbox.push.apple.com';
        
        $url = $endpoint . '/3/device/' . $deviceToken;
        
        // Headers untuk request
        $headers = [
            'apns-topic: ' . $this->bundleId,
            'apns-push-type: alert',
            'authorization: bearer ' . $authToken,
            'content-type: application/json'
        ];
        
        // Tambahkan options tambahan
        if (isset($options['priority'])) {
            $headers[] = 'apns-priority: ' . $options['priority'];
        }
        
        if (isset($options['expiration'])) {
            $headers[] = 'apns-expiration: ' . $options['expiration'];
        }
        
        // Kirim request menggunakan cURL
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($customPayload));
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTP_VERSION, CURL_HTTP_VERSION_2_0);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        
        curl_close($ch);
        
        return [
            'success' => $httpCode === 200,
            'http_code' => $httpCode,
            'response' => $response,
            'error' => $error,
            'payload' => $customPayload,
            'headers' => $headers
        ];
    }
    
    /**
     * Test koneksi ke APNs
     */
    public function testConnection()
    {
        try {
            $authToken = $this->generateAuthToken();
            return [
                'success' => true,
                'message' => 'Koneksi ke APNs berhasil',
                'auth_token' => substr($authToken, 0, 50) . '...'
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => 'Gagal koneksi ke APNs: ' . $e->getMessage()
            ];
        }
    }
}
